"use client"

import * as React from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell
} from "@/components/ui/table"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem
} from "@/components/ui/select"
import { Printer, PlusCircle } from "lucide-react"

type Supplier = { id: string; name: string }
type Bill = {
  id: string
  no: string
  date: string
  supplierId: string
  supplierName: string
  total: number
  paid: number
  status: "OPEN" | "PAID" | "APPROVED"
  createdBy: string
}

const suppliersSeed: Supplier[] = [
  { id: "s1", name: "Tempcon Trading" },
  { id: "s2", name: "Nice Touch" }
]

export default function BillsPage() {
  const router = useRouter()
  const [dateFrom, setDateFrom] = React.useState(new Date().toISOString().slice(0, 10))
  const [dateTo, setDateTo] = React.useState(new Date().toISOString().slice(0, 10))
  const [supplierFilter, setSupplierFilter] = React.useState("all")
  const [q, setQ] = React.useState("")
  const [rows] = React.useState<Bill[]>([]) // Add rows via API

  const filtered = rows.filter(b => {
    if (supplierFilter !== "all" && b.supplierId !== supplierFilter) return false
    if (dateFrom && b.date < dateFrom) return false
    if (dateTo && b.date > dateTo) return false
    if (q && !(`${b.no} ${b.supplierName}`.toLowerCase().includes(q.toLowerCase()))) return false
    return true
  })

  return (
    <div className="w-full p-2 space-y-6">
      {/* Filters + New Bill Button */}
      <div className="flex flex-wrap items-end justify-between gap-3">
        <div className="flex flex-wrap gap-3 flex-grow">
          <Field label="From" className="w-[150px]">
            <Input type="date" className="h-8 text-sm" value={dateFrom} onChange={e => setDateFrom(e.target.value)} />
          </Field>
          <Field label="To" className="w-[150px]">
            <Input type="date" className="h-8 text-sm" value={dateTo} onChange={e => setDateTo(e.target.value)} />
          </Field>
          <Field label="Supplier" className="w-[180px]">
            <Select value={supplierFilter} onValueChange={setSupplierFilter}>
              <SelectTrigger className="h-8 text-sm">
                <SelectValue placeholder="All" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All</SelectItem>
                {suppliersSeed.map(s => (
                  <SelectItem key={s.id} value={s.id}>{s.name}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </Field>
        </div>
        <Button className="h-8 text-sm gap-1" onClick={() => router.push("/inventory/bills/new")}>
          <PlusCircle className="size-4" /> New
        </Button>
      </div>


      {/* Title + Toolbar */}
      <div className="flex flex-wrap items-center justify-between gap-4">
        <h2 className="text-2xl font-semibold">All Bills</h2>
        <div className="flex items-center gap-3">
          <Button variant="outline" className="h-9" onClick={() => window.print()}>
            <Printer className="size-4 mr-2" /> Print
          </Button>
          <div className="flex items-center gap-2">
            <span className="text-sm">Search:</span>
            <Input className="h-9 w-64" value={q} onChange={e => setQ(e.target.value)} />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-hidden rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Bill No</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Supplier</TableHead>
              <TableHead>Total Amount</TableHead>
              <TableHead>Paid Amount</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead className="w-28 text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filtered.length ? (
              filtered.map(b => (
                <TableRow key={b.id}>
                  <TableCell>{b.no}</TableCell>
                  <TableCell>{b.date}</TableCell>
                  <TableCell>{b.supplierName}</TableCell>
                  <TableCell>{b.total.toFixed(2)}</TableCell>
                  <TableCell>{b.paid.toFixed(2)}</TableCell>
                  <TableCell>{b.status}</TableCell>
                  <TableCell>{b.createdBy}</TableCell>
                  <TableCell className="text-right">
                    <Button variant="outline" size="sm">View</Button>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={8} className="text-center text-muted-foreground">
                  No data available in table
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}

function Field({ label, className, children }: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={className}>
      <label className="mb-1 block text-sm font-semibold">{label}</label>
      {children}
    </div>
  )
}
