"use client";

import * as React from "react";
import { useRouter } from "next/navigation";
import { useTranslations } from "next-intl";
import { Plus, Trash2, Settings2, Save, Brush, BrushCleaning, RefreshCw } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { UnitsTable } from "@/components/table/components/units-table";
import { InventoryService } from "@/lib/api/services/inventoryService";
import type { CreateInventoryItemData } from "@/lib/types/inventory";
import { Controller, useFieldArray, useForm } from "react-hook-form";
import SubmitBtnClient from "@/components/submit-client-button";
import { useEffect } from "react";
import { TableSkeleton } from "@/components/ui/table-skeleton";
import { toast } from "sonner";
import { ItemRow } from "@/components/inventory/itemCreateRow";


interface InventoryItem {
  item_name: string;
  category_id: string;
  item_type: string;
  unit_id: string;
  cost_price: string;
  barcode: string;
  reorder_level: string;
  opening_stock: string;
  batch_number: string;
  expiry_date: string | null;
  location: string,
  department_id: string,
  min_stock: string
  unit_conversions: []
}


interface GeneralFormValues {
  items: InventoryItem[];
}

export default function CreateItemPage() {
  const router = useRouter();
  const t = useTranslations("ItemForm");
  const [isSubmitting, setIsSubmitting] = React.useState<boolean>(false);
  const [loading, setLoading] = React.useState<boolean>(true);
  const [createData, setCreateData] = React.useState<any>();
  const hasRun = React.useRef(false);

  const methods = useForm<GeneralFormValues>({
    defaultValues: {
      items: [], // start empty
    },
  });

  async function createItem() {
    try {
      const createData = await InventoryService.createItem();
      debugger;
      setCreateData(createData);
    } catch (error: any) {
      const data = error.response?.data;
      if (data?.errors) {
        // Flatten all messages
        const messages = Object.values(data.errors).flat();
        messages.forEach((msg) => toast.error(String(msg)));
      } else if (data?.message) {
        toast.error(data.message);
      } else {
        toast.error("Failed to get item creation details");
      }
    } finally {
      setLoading(false);
    }

  }

  useEffect(() => {
    if (hasRun.current) return; // Skip if already run
    hasRun.current = true;
    createItem();
    appendItemRow();
  }, []);

  const { control, register, reset, handleSubmit } = methods;
  const { fields, append, update, remove, } = useFieldArray({
    control,
    name: "items",
  });



  async function onSave(data: any) {
    setIsSubmitting(true);
    try {
      const formData = new FormData();

      await InventoryService.storeItem(data);
      router.push("/inventory/items");
      toast.success("Item added successfully");
    } catch (error: any) {
      const data = error.response?.data;
      if (data?.errors) {
        // Flatten all messages
        const messages = Object.values(data.errors).flat();
        messages.forEach((msg) => toast.error(String(msg)));
      } else if (data?.message) {
        toast.error(data.message);
      } else {
        toast.error("Failed to create item");
      }
    } finally {
      setIsSubmitting(false);
    }
  }

  const appendItemRow = () => {
    append({
      item_name: "",
      category_id: "",
      item_type: "",
      unit_id: "",
      cost_price: "0",
      barcode: "",
      reorder_level: "0",
      opening_stock: "0",
      batch_number: "",
      expiry_date: null,
      location: "",
      unit_conversions: [],
      department_id: "",
      min_stock: "0"
    })
  }


  return (
    <div className="w-full p-2 space-y-3">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-semibold">{t("addNewItemTitle")}</h1>
        <Button onClick={appendItemRow} size="sm">
          <Plus className="mr-1 size-4" />
          {t("add")}
        </Button>
      </div>

      <div className="space-y-3">
        {loading ? (
          <TableSkeleton columns={6} rows={1} />) : (
          <div>
            {fields.map((field, i) => (
              <ItemRow
                key={field.id}
                i={i}
                control={control}
                register={register}
                remove={remove}
                methods={methods}
                createData={createData}
                reset={reset}
              />
            ))}
          </div>
        )}
      </div>

      {/* Footer */}
      <div className="sticky bottom-0 flex justify-end gap-2 border-t bg-background p-3">
        <Button variant="outline" size="sm" onClick={() => router.back()}>
          {t("cancel")}
        </Button>
        <SubmitBtnClient
          isLoading={isSubmitting}
          onClick={handleSubmit(onSave)}
        >
          <Save className=" h-4 w-4" />
          {t("saveItems")}
        </SubmitBtnClient>
      </div>
    </div>
  );
}


