"use client";

import { useCallback, useEffect, useState } from "react";
import {
  MoreHorizontal,
  Edit,
  Copy,
  Trash2,
  ChevronLeft,
  ChevronRight,
  ChevronsLeft,
  ChevronsRight,
  Download,
  Link,
  Plus,
  Printer,
  RefreshCw,
  History,
  Eye,
  Grid,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { TableSkeleton } from "@/components/ui/table-skeleton";
import { ConfirmDialog } from "@/components/alert-delete";
import { LinkAsButton } from "@/components/ui/link-button";
import { Input } from "@/components/ui/input";
import { useTranslations } from "next-intl";
import { useGenericTable } from "@/lib/hooks/useGenericTable";
import { API_ENDPOINTS } from "@/lib/api/endpoints";
import { Card, CardHeader, CardTitle, CardDescription, CardContent } from "@/components/ui/card";
import { Label } from "@radix-ui/react-label";
import { DeleteButton } from "@/components/DeleteButton";
import { DynamicModal } from "@/components/common/dynamicModal";
import { useForm } from "react-hook-form";


interface Column {
  key: string;
  label: string;
  className?: string;
}

interface PaginationProps {
  currentPage: number;
  totalPages: number;
  totalItems: number;
  itemsPerPage: number;
}


interface GenericTableProps {

  onPageChange: (page: number) => void;
  onRetry?: () => void;
  onEdit?: (row: any) => void;
  onClone?: (row: any) => void;
  onDelete?: (id: string) => void;
  deleteConfirmText?: { title: string; description: string };
  permissions?: {
    edit?: boolean;
    clone?: boolean;
    delete?: boolean;
  };
}

export default function ItemPage({

  onPageChange,
  onRetry,
  onEdit,
  onClone,
  onDelete,
  deleteConfirmText = {
    title: "Are you sure?",
    description: "This action cannot be undone.",
  },
  permissions = { edit: true, clone: true, delete: true },
}: GenericTableProps) {
  const url = API_ENDPOINTS.INVENTORY.ITEMS;

  const { data, loading, error, pagination, fetchData, save, deleteItem } = useGenericTable({ url });
  const [deleteModal, setDeleteModal] = useState(false);
  const [createModal, setCreateModal] = useState(false);
  const [selectedId, setSelectedId] = useState<string | null>(null);
  const [searchFilters, setSearchFilters] = useState<any>([]);
  const t = useTranslations("InventoryTable");
  const title = t('itemListTitle');
  const columns = [
    { key: "id", label: t('no') },
    { key: "barcode", label: t('barcode') },
    { key: "item_name", label: t('item_name') },
    { key: "available_stock", label: t('availableStock') },
    { key: "base_unit", label: t('unitOfMeasure') },
    { key: "category", label: t('itemCategory') },
    { key: "item_type", label: t('type') },
    { key: "expiry_date", label: t('expiryDate') },
  ];

  const s = useTranslations("ItemForm");

  const addStock = [
    { label: s('quantity'), data: { type: "text", props: "", value: "", key: "quantity" } },
    { label: s('cost'), data: { type: "text", props: "", value: "", key: "cost" } },
    { label: s('batch'), data: { type: "text", props: "", value: "", key: "batch" } },
    { label: s('expiryDate'), data: { type: "date", props: "", value: "", key: "expiryDate" } },
    { label: s('location'), data: { type: "text", props: "", value: "", key: "location" } },
    { label: s('notes'), data: { type: "textarea", props: "", value: "", key: "notes" } }
  ];
  const titleStock = {
    title: "Add Stock",
    desc: "Add new item stock"
  }

  const handleDelete = (id: string) => {
    setDeleteModal(true);
    setSelectedId(id);
  };

  const methods = useForm<any>();
  const { control, register, reset, handleSubmit } = methods;

  const buildSearchParams =
    (page: number = 1) => {
      const params = new URLSearchParams({
        page: page.toString(),
        per_page: "10",
      });

      if (searchFilters?.barcode) {
        params.set("barcode", searchFilters.barcode);
      }
      if (searchFilters?.name) {
        params.set("item_name", searchFilters.name);
      }
      return params.toString();
    };

  // Fetch user when search filters change
  useEffect(() => {
    debugger;
    const params = buildSearchParams(1)
    fetchData(params)
  }, [searchFilters])

  // Handle pagination
  const handlePageChange = ((newPage: number) => {
    debugger;
    const params = buildSearchParams(newPage)
    fetchData(params);
  })

  const handleDeleteConfirm = () => {
    deleteItem(selectedId || '');
  };

  const handleSave = async (data: any) => {
    debugger;
    save(data);
  };

  const handleSearch = (field: string, value: string) => {
    setSearchFilters((prev: any) => ({ ...prev, [field]: value }))
  }

  // === Error ===
  if (error) {
    return (
      <div className="p-6 text-center text-red-600">
        <p>{error}</p>
        {onRetry && (
          <Button variant="outline" onClick={onRetry} className="mt-2">
            Retry
          </Button>
        )}
      </div>
    );
  }

  return (
    <div className="w-full p-2 space-y-6">
      {/* Top bar */}


      <Card className="mb-6">
        <CardHeader>
          <CardTitle className="text-lg">
            <div className="flex flex-wrap items-center justify-between gap-3">
              <div className="flex items-center gap-2">
                {title && (
                  <h2 className="text-3xl font-semibold text-gray-700">{title}</h2>
                )}
              </div>

              <div className="flex items-center gap-2">
                <Button asChild>
                  <LinkAsButton href="/inventory/items/create">
                    <Plus className="mr-2 size-4" />
                    {t("add")}
                  </LinkAsButton>
                </Button>
                <Button variant="outline">
                  <Printer className="mr-2 size-4" />
                  {t("print")}
                </Button>
                <Button variant="outline">
                  <Download className="mr-2 size-4" />
                  {t("export")}
                </Button>
              </div>
            </div>
          </CardTitle>
          <CardDescription>{t("filter.filter_desc")}</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <div className="space-y-2">
              <Label htmlFor="barcode">{t("filter.barcode")}</Label>
              <Input
                id="name"
                placeholder={`${t("filter.search_by")} ${t(
                  "filter.barcode"
                )}`}
                value={searchFilters.barcode || ""}
                onChange={(e) => handleSearch("barcode", e.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="name">{t("filter.name_label")}</Label>
              <Input
                id="name"
                placeholder={`${t("filter.search_by")} ${t(
                  "filter.name_label"
                )}`}
                value={searchFilters.name || ""}
                onChange={(e) => handleSearch("name", e.target.value)}
              />
            </div>
          </div>
        </CardContent>
      </Card>



      <Card>
        <CardContent className="p-0">
          <div className="space-y-4 p-1">


            {/* Pagination Info */}
            <div className="flex items-center justify-between text-sm text-gray-600">
              <span>
                {pagination && (
                  <div>
                    Showing{" "}
                    {(pagination.currentPage - 1) * pagination.itemsPerPage + 1} to{" "}
                    {Math.min(
                      pagination.currentPage * pagination.itemsPerPage,
                      pagination.totalItems
                    )}{" "}
                    of {pagination.totalItems} entries
                  </div>
                )}
              </span>
            </div>

            {loading ? (<TableSkeleton columns={9} rows={16} />) : (
              <div className="overflow-hidden ">
                <Table>
                  <TableHeader>
                    <TableRow>
                      {columns?.map((col) => (
                        <TableHead key={col.key}>
                          {col.label}
                        </TableHead>
                      ))}
                      <TableHead className="w-12 text-right">Actions</TableHead>
                    </TableRow>
                  </TableHeader>

                  <TableBody>
                    <TableRow>
                      {data?.length === 0 && (
                        <TableCell
                          colSpan={columns.length + 1}
                          className="text-center py-6"
                        >
                          No records found
                        </TableCell>
                      )}
                    </TableRow>


                    {data?.map((row: any) => (
                      <TableRow key={row?.id} className="hover:bg-gray-50">
                        <TableCell>{row?.id ?? "-"}</TableCell>
                        <TableCell>{row?.barcode ?? "-"}</TableCell>
                        <TableCell>{row?.item_name ?? "-"}</TableCell>
                        <TableCell>{row?.available_stock ?? "-"}</TableCell>
                        <TableCell>{row?.base_unit_name ?? "-"}</TableCell>
                        <TableCell>{row?.category ?? "-"}</TableCell>
                        <TableCell>{row?.item_type ?? "-"}</TableCell>
                        <TableCell>{row?.expiry_date ?? "-"}</TableCell>
                        <TableCell className="text-right">
                          <DropdownMenu modal={false}>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" className="h-8 w-8 p-0">
                                <MoreHorizontal className="h-4 w-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem
                                onClick={() => onClone?.(row)}
                                className="cursor-pointer"
                              >
                                <Eye className="mr-2 h-4 w-4" />
                                View
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                className="cursor-pointer"
                                onClick={() => setCreateModal(true)}
                              >
                                <Grid className="mr-2 h-4 w-4" />
                                Add Stock
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                className="cursor-pointer"
                              >
                                <History className="mr-2 h-4 w-4" />
                                Histoy
                              </DropdownMenuItem>
                              {permissions.edit && (
                                <DropdownMenuItem
                                  onClick={() => onEdit?.(row)}
                                  className="cursor-pointer"
                                >
                                  <Edit className="mr-2 h-4 w-4" />
                                  Edit Item
                                </DropdownMenuItem>
                              )}
                              {permissions.delete && (
                                <>
                                  <DropdownMenuSeparator />
                                  <DropdownMenuItem
                                    asChild
                                    onSelect={(e) => e.preventDefault()} // Prevent Dropdown auto-close
                                  >
                                    <div onClick={(e) => e.stopPropagation()}>
                                      <DeleteButton key={row.id} onConfirm={() => deleteItem(row.id)} />
                                      Delete
                                    </div>
                                  </DropdownMenuItem>
                                </>
                              )}
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </TableCell>
                      </TableRow>
                    ))
                    }
                  </TableBody>
                </Table>
              </div>
            )}

            {/* Pagination */}
            {pagination?.totalPages > 1 && (
              <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">
                  Page {pagination.currentPage} of {pagination.totalPages}
                </div>
                <div className="flex items-center space-x-2">
                  <Button
                    variant="outline"
                    className="hidden h-8 w-8 p-0 lg:flex"
                    onClick={() => handlePageChange(1)}
                    disabled={pagination.currentPage === 1}
                  >
                    <ChevronsLeft className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handlePageChange(pagination.currentPage - 1)}
                    disabled={pagination.currentPage === 1}
                  >
                    <ChevronLeft className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handlePageChange(pagination.currentPage + 1)}
                    disabled={pagination.currentPage === pagination.totalPages}
                  >
                    <ChevronRight className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="outline"
                    className="hidden h-8 w-8 p-0 lg:flex"
                    onClick={() => handlePageChange(pagination.totalPages)}
                    disabled={pagination.currentPage === pagination.totalPages}
                  >
                    <ChevronsRight className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            )}

            {/* Confirm Dialog */}
            <ConfirmDialog
              key={2}
              isOpen={deleteModal}
              setIsOpen={setDeleteModal}
              title={deleteConfirmText.title}
              description={deleteConfirmText.description}
              onConfirm={handleDeleteConfirm}
            />
            {/* <DynamicModal
              key={3}
              title={titleStock}
              isOpen={createModal}
              control={control}
              loading={loading}
              openChange={setCreateModal}
              onConfirm={() => handleSave} fieldset={addStock} columns={2} /> */}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
