"use client";

import * as React from "react";
import { useTranslations } from "next-intl";
import { Plus, Trash2, Send } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select";
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell,
} from "@/components/ui/table";

// --- Types ---
type Line = {
  id: string;
  item?: string;
  description?: string;
  unit?: string;
  qty: number | "";
  price: number | "";
  discount: number | "";
  tax: number | "";
};

type Supplier = {
  id: string;
  name: string;
  email?: string;
  phone?: string;
};

type Props = {
  initialData?: any;
  onSave?: () => void;
};

// --- Seed Data (Replace with API) ---
const suppliers: Supplier[] = [
  { id: "s1", name: "Acme Supplies", email: "sales@acme.com", phone: "+1 555 123" },
  { id: "s2", name: "Global Med", email: "orders@globalmed.com", phone: "+1 555 456" },
];

const items = [
  { id: "i1", name: "Surgical Gloves", defaultUnit: "box", defaultPrice: 12.5, desc: "Powder-free, latex" },
  { id: "i2", name: "Syringe 5ml", defaultUnit: "pc", defaultPrice: 0.35, desc: "Luer lock" },
  { id: "i3", name: "Alcohol Swab", defaultUnit: "pk", defaultPrice: 2.1, desc: "Isopropyl 70%" },
];

const units = [
  { id: "pc", label: "PC" },
  { id: "box", label: "BOX" },
  { id: "pk", label: "PK" },
];

export default function CreatePurchaseOrderPage({ initialData, onSave }: Props) {
  const t = useTranslations("PurchaseOrders");

  const [poNumber, setPoNumber] = React.useState(initialData?.poNumber || "PO-000123");
  const [supplierId, setSupplierId] = React.useState(initialData?.supplierId || "s1");
  const [supplierEmail, setSupplierEmail] = React.useState(initialData?.supplierEmail || "");
  const [supplierPhone, setSupplierPhone] = React.useState(initialData?.supplierPhone || "");
  const [poDate, setPoDate] = React.useState(initialData?.poDate || new Date().toISOString().slice(0, 10));
  const [expectedDate, setExpectedDate] = React.useState(initialData?.expectedDate || new Date().toISOString().slice(0, 10));
  const [currency, setCurrency] = React.useState(initialData?.currency || "QAR");
  const [terms, setTerms] = React.useState(initialData?.terms || "net30");
  const [status, setStatus] = React.useState(initialData?.status || "draft");
  const [deliverTo, setDeliverTo] = React.useState(initialData?.deliverTo || "main");
  const [shipping, setShipping] = React.useState(initialData?.charges?.shipping?.toString() || "0");
  const [other, setOther] = React.useState(initialData?.charges?.other?.toString() || "0");
  const [notes, setNotes] = React.useState(initialData?.notes || "");
  const [lines, setLines] = React.useState<Line[]>(
    initialData?.lines || [{ id: crypto.randomUUID(), qty: "", price: "", discount: "", tax: "" }]
  );

  // Auto-fill contact
  React.useEffect(() => {
    const s = suppliers.find((x) => x.id === supplierId);
    if (s) {
      setSupplierEmail(s.email || "");
      setSupplierPhone(s.phone || "");
    }
  }, [supplierId]);

  const addLine = () => setLines((prev) => [
    ...prev,
    { id: crypto.randomUUID(), qty: "", price: "", discount: "", tax: "" },
  ]);

  const removeLine = (id: string) =>
    setLines((prev) => prev.filter((l) => l.id !== id));

  const patchLine = (id: string, patch: Partial<Line>) =>
    setLines((prev) => prev.map((l) => (l.id === id ? { ...l, ...patch } : l)));

  const onSelectItem = (lineId: string, itemId: string) => {
    const it = items.find((x) => x.id === itemId);
    if (it) {
      patchLine(lineId, {
        item: itemId,
        description: it.desc,
        unit: it.defaultUnit,
        price: it.defaultPrice,
      });
    }
  };

  const calc = React.useMemo(() => {
    let subtotal = 0,
      discountTotal = 0,
      taxTotal = 0;

    lines.forEach((l) => {
      const qty = Number(l.qty || 0);
      const price = Number(l.price || 0);
      const disc = Number(l.discount || 0);
      const tax = Number(l.tax || 0);

      const base = qty * price;
      const lineDisc = (disc / 100) * base;
      const afterDisc = base - lineDisc;
      const lineTax = (tax / 100) * afterDisc;

      subtotal += base;
      discountTotal += lineDisc;
      taxTotal += lineTax;
    });

    const shippingNum = Number(shipping || 0);
    const otherNum = Number(other || 0);
    const grand = subtotal - discountTotal + taxTotal + shippingNum + otherNum;

    return { subtotal, discountTotal, taxTotal, shipping: shippingNum, other: otherNum, grand };
  }, [lines, shipping, other]);

  function payload(submit: boolean) {
    return {
      poNumber,
      supplierId,
      supplierEmail,
      supplierPhone,
      poDate,
      expectedDate,
      currency,
      terms,
      status: submit ? "sent" : status,
      deliverTo,
      lines,
      charges: { shipping: Number(shipping || 0), other: Number(other || 0) },
      notes,
      totals: calc,
    };
  }

  function onSaveDraft() {
    console.log("SAVE DRAFT", payload(false));
  }

  function onSubmit() {
    console.log("SUBMIT", payload(true));
    onSave?.();
  }

  return (
    <div className="w-full p-4 md:p-6 space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-xl md:text-2xl font-semibold tracking-tight">
          {initialData ? t("edit") : t("newPO")}
        </h1>
        <Button className="h-9" onClick={onSubmit}>
          <Send className="mr-2 size-4" />
          {t("submitSend")}
        </Button>
      </div>

      {/* Header Grid - Row A */}
      {/* Header Form Section */}
      <div className="rounded-md border bg-card p-4 space-y-4">
        <div className="grid grid-cols-12 gap-x-3 gap-y-4">
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("poNumber")}>
            <Input
              className="h-9 w-full"
              value={poNumber}
              onChange={(e) => setPoNumber(e.target.value)}
            />
          </Field>
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("poDate")}>
            <Input
              type="date"
              className="h-9 w-full"
              value={poDate}
              onChange={(e) => setPoDate(e.target.value)}
            />
          </Field>
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("expectedDate")}>
            <Input
              type="date"
              className="h-9 w-full"
              value={expectedDate}
              onChange={(e) => setExpectedDate(e.target.value)}
            />
          </Field>
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("status")}>
            <Select value={status} onValueChange={(v) => setStatus(v as any)}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="draft">{t("statusDraft")}</SelectItem>
                <SelectItem value="sent">{t("statusSent")}</SelectItem>
                <SelectItem value="partial">{t("statusPartial")}</SelectItem>
                <SelectItem value="received">{t("statusReceived")}</SelectItem>
                <SelectItem value="closed">{t("statusClosed")}</SelectItem>
              </SelectContent>
            </Select>
          </Field>
        </div>
        
        <div className="grid grid-cols-12 gap-x-3 gap-y-4">
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("currency")}>
            <Select value={currency} onValueChange={setCurrency}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {["USD", "EUR", "QAR", "AED", "SAR"].map((c) => (
                  <SelectItem key={c} value={c}>{c}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </Field>
              
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("terms")}>
            <Select value={terms} onValueChange={setTerms}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="net15">NET 15</SelectItem>
                <SelectItem value="net30">NET 30</SelectItem>
                <SelectItem value="prepaid">{t("termsPrepaid")}</SelectItem>
              </SelectContent>
            </Select>
          </Field>
              
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("supplier")}>
            <Select value={supplierId} onValueChange={setSupplierId}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue placeholder={t("selectSupplier")} />
              </SelectTrigger>
              <SelectContent>
                {suppliers.map((s) => (
                  <SelectItem key={s.id} value={s.id}>{s.name}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </Field>
              
          <Field className="col-span-12 sm:col-span-6 md:col-span-3" label={t("deliverTo")}>
            <Select value={deliverTo} onValueChange={setDeliverTo}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="main">{t("storeMain")}</SelectItem>
                <SelectItem value="room1">{t("storeRoom1")}</SelectItem>
              </SelectContent>
            </Select>
          </Field>
        </div>
              
        <div className="grid grid-cols-12 gap-x-3 gap-y-4">
          <Field className="col-span-12 sm:col-span-6" label={t("supplierEmail")}>
            <Input
              className="h-9 w-full"
              value={supplierEmail}
              onChange={(e) => setSupplierEmail(e.target.value)}
            />
          </Field>
          <Field className="col-span-12 sm:col-span-6" label={t("supplierPhone")}>
            <Input
              className="h-9 w-full"
              value={supplierPhone}
              onChange={(e) => setSupplierPhone(e.target.value)}
            />
          </Field>
        </div>
      </div>


      {/* Line Items */}
      <div className="rounded-md border overflow-hidden">
        <div className="flex items-center justify-between bg-card p-3">
          <h2 className="text-sm font-medium">{t("itemsSection")}</h2>
          <Button variant="outline" size="sm" onClick={addLine} className="h-8">
            <Plus className="mr-2 size-4" />
            {t("addRow")}
          </Button>
        </div>

        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="min-w-[160px]">{t("item")}</TableHead>
                <TableHead className="min-w-[200px]">{t("description")}</TableHead>
                <TableHead className="w-24">{t("unit")}</TableHead>
                <TableHead className="w-24">{t("qty")}</TableHead>
                <TableHead className="w-28">{t("unitPrice")}</TableHead>
                <TableHead className="w-24">{t("discount")} %</TableHead>
                <TableHead className="w-24">{t("tax")} %</TableHead>
                <TableHead className="w-32 text-right">{t("lineTotal")}</TableHead>
                <TableHead className="w-12" />
              </TableRow>
            </TableHeader>
            <TableBody>
              {lines.map((l) => {
                const qty = Number(l.qty || 0);
                const price = Number(l.price || 0);
                const disc = Number(l.discount || 0);
                const tax = Number(l.tax || 0);
                const base = qty * price;
                const afterDisc = base - (disc / 100) * base;
                const lineTotal = afterDisc + (tax / 100) * afterDisc;

                return (
                  <TableRow key={l.id}>
                    <TableCell>
                      <Select value={l.item || ""} onValueChange={(v) => onSelectItem(l.id, v)}>
                        <SelectTrigger className="h-9 w-full">
                          <SelectValue placeholder={t("selectItem")} />
                        </SelectTrigger>
                        <SelectContent>
                          {items.map((it) => (
                            <SelectItem key={it.id} value={it.id}>{it.name}</SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </TableCell>
                    <TableCell>
                      <Input
                        className="h-9"
                        value={l.description || ""}
                        onChange={(e) => patchLine(l.id, { description: e.target.value })}
                      />
                    </TableCell>
                    <TableCell>
                      <Select value={l.unit || ""} onValueChange={(v) => patchLine(l.id, { unit: v })}>
                        <SelectTrigger className="h-9 w-full">
                          <SelectValue placeholder={t("selectUnit")} />
                        </SelectTrigger>
                        <SelectContent>
                          {units.map((u) => (
                            <SelectItem key={u.id} value={u.id}>{u.label}</SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </TableCell>
                    <TableCell>
                      <Input
                        className="h-9"
                        type="number"
                        value={l.qty}
                        onChange={(e) =>
                          patchLine(l.id, { qty: e.target.value === "" ? "" : Number(e.target.value) })
                        }
                      />
                    </TableCell>
                    <TableCell>
                      <Input
                        className="h-9"
                        type="number"
                        value={l.price}
                        onChange={(e) =>
                          patchLine(l.id, { price: e.target.value === "" ? "" : Number(e.target.value) })
                        }
                      />
                    </TableCell>
                    <TableCell>
                      <Input
                        className="h-9"
                        type="number"
                        value={l.discount}
                        onChange={(e) =>
                          patchLine(l.id, { discount: e.target.value === "" ? "" : Number(e.target.value) })
                        }
                      />
                    </TableCell>
                    <TableCell>
                      <Input
                        className="h-9"
                        type="number"
                        value={l.tax}
                        onChange={(e) =>
                          patchLine(l.id, { tax: e.target.value === "" ? "" : Number(e.target.value) })
                        }
                      />
                    </TableCell>
                    <TableCell className="text-right tabular-nums">
                      {lineTotal.toFixed(2)}
                    </TableCell>
                    <TableCell className="text-right">
                      <Button size="icon" variant="outline" onClick={() => removeLine(l.id)}>
                        <Trash2 className="size-4" />
                      </Button>
                    </TableCell>
                  </TableRow>
                );
              })}
              {lines.length === 0 && (
                <TableRow>
                  <TableCell colSpan={9} className="text-center text-muted-foreground">
                    {t("noRows")}
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      {/* Totals & Notes */}
      <div className="grid grid-cols-12 gap-4">
        <div className="col-span-12 lg:col-span-7">
          <label className="mb-1 block text-xs text-muted-foreground">{t("notes")}</label>
          <Textarea value={notes} onChange={(e) => setNotes(e.target.value)} className="min-h-[120px]" />
        </div>
        <div className="col-span-12 lg:col-span-5">
          <div className="rounded-md border p-4 space-y-3">
            <Row label={t("subtotal")} value={calc.subtotal} />
            <Row label={t("discountTotal")} value={calc.discountTotal} />
            <Row label={t("taxTotal")} value={calc.taxTotal} />

            <div className="flex items-center justify-between">
              <span className="text-sm text-muted-foreground">{t("shipping")}</span>
              <Input
                className="h-8 w-32 text-right"
                type="number"
                value={shipping}
                onChange={(e) => setShipping(e.target.value)}
              />
            </div>
            <div className="flex items-center justify-between">
              <span className="text-sm text-muted-foreground">{t("otherCharges")}</span>
              <Input
                className="h-8 w-32 text-right"
                type="number"
                value={other}
                onChange={(e) => setOther(e.target.value)}
              />
            </div>
            <div className="mt-2 border-t pt-2 flex items-center justify-between">
              <span className="font-medium">{t("grandTotal")}</span>
              <span className="font-semibold tabular-nums">
                {calc.grand.toFixed(2)} {currency}
              </span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}

/** Simple labeled field helper */

function Field({ label, className, children }: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={className}>
      <label className="mb-1 block text-xs font-medium text-muted-foreground">{label}</label>
      {children}
    </div>
  );
}
function Row({ label, value }: { label: string; value: number }) {
  return (
    <div className="flex items-center justify-between">
      <span className="text-sm text-muted-foreground">{label}</span>
      <span className="tabular-nums">{value.toFixed(2)}</span>
    </div>
  );
}