"use client";

import * as React from "react";
import Link from "next/link";
import { useTranslations } from "next-intl";
import { Plus, Printer, FileDown, Search } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select, SelectTrigger, SelectValue, SelectContent, SelectItem
} from "@/components/ui/select";
import {
  Table, TableHeader, TableRow, TableHead, TableBody, TableCell
} from "@/components/ui/table";

// Demo data
type PO = {
  id: string;
  poNumber: string;
  supplier: string;
  date: string;
  expected: string;
  status: "draft" | "sent" | "partial" | "closed";
  total: number;
  currency: string;
};

const demo: PO[] = [
  {
    id: "1",
    poNumber: "PO-000123",
    supplier: "Acme Supplies",
    date: "2025-08-01",
    expected: "2025-08-05",
    status: "draft",
    total: 523.4,
    currency: "QAR",
  },
  {
    id: "2",
    poNumber: "PO-000124",
    supplier: "Global Med",
    date: "2025-08-02",
    expected: "2025-08-07",
    status: "sent",
    total: 1299.0,
    currency: "QAR",
  },
  {
    id: "3",
    poNumber: "PO-000125",
    supplier: "HealthX",
    date: "2025-08-03",
    expected: "2025-08-10",
    status: "partial",
    total: 210.75,
    currency: "QAR",
  },
];

export default function PurchaseOrdersListPage() {
  const t = useTranslations("PurchaseOrders");

  const [q, setQ] = React.useState("");
  const [status, setStatus] = React.useState<"all" | "draft" | "sent" | "partial" | "closed">("all");
  const [from, setFrom] = React.useState<string>("");
  const [to, setTo] = React.useState<string>("");

  const rows = React.useMemo(() => {
    return demo.filter((po) => {
      const matchesText =
        !q ||
        po.poNumber.toLowerCase().includes(q.toLowerCase()) ||
        po.supplier.toLowerCase().includes(q.toLowerCase());
      const matchesStatus = status === "all" || po.status === status;
      const afterFrom = !from || po.date >= from;
      const beforeTo = !to || po.date <= to;
      return matchesText && matchesStatus && afterFrom && beforeTo;
    });
  }, [q, status, from, to]);

  return (
    <div className="w-full max-w-none px-4 md:px-6 lg:px-8 py-4 space-y-6">
      {/* Header */}
      <div className="flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
        <h1 className="text-2xl font-semibold tracking-tight">{t("newPO")}</h1>
        <div className="flex gap-2">
          <Button asChild>
            <Link href="/inventory/purchase-orders/create">
              <Plus className="mr-2 size-4" />
              {t("createPO")}
            </Link>
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="rounded-md border bg-card p-4">
        <div className="grid grid-cols-12 gap-3">
          <div className="col-span-12 lg:col-span-4">
            <label className="mb-1 block text-xs text-muted-foreground">{t("search")}</label>
            <div className="relative">
              <Search className="absolute left-2 top-1/2 -translate-y-1/2 size-4 text-muted-foreground" />
              <Input
                className="pl-8 h-9"
                placeholder={t("searchPlaceholder")}
                value={q}
                onChange={(e) => setQ(e.target.value)}
              />
            </div>
          </div>

          <div className="col-span-6 lg:col-span-2">
            <label className="mb-1 block text-xs text-muted-foreground">{t("status")}</label>
            <Select value={status} onValueChange={(v) => setStatus(v as typeof status)}>
              <SelectTrigger className="h-9 w-full">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">{t("all")}</SelectItem>
                <SelectItem value="draft">{t("statusDraft")}</SelectItem>
                <SelectItem value="sent">{t("statusSent")}</SelectItem>
                <SelectItem value="partial">{t("statusPartial")}</SelectItem>
                <SelectItem value="closed">{t("statusClosed")}</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="col-span-6 lg:col-span-2">
            <label className="mb-1 block text-xs text-muted-foreground">{t("dateFrom")}</label>
            <Input className="h-9" type="date" value={from} onChange={(e) => setFrom(e.target.value)} />
          </div>

          <div className="col-span-6 lg:col-span-2">
            <label className="mb-1 block text-xs text-muted-foreground">{t("dateTo")}</label>
            <Input className="h-9" type="date" value={to} onChange={(e) => setTo(e.target.value)} />
          </div>
        </div>
      </div>

      {/* Table */}
      <div className="overflow-x-auto rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="min-w-[120px]">{t("poNumber")}</TableHead>
              <TableHead className="min-w-[200px]">{t("supplier")}</TableHead>
              <TableHead className="min-w-[120px]">{t("poDate")}</TableHead>
              <TableHead className="min-w-[140px]">{t("expectedDate")}</TableHead>
              <TableHead className="min-w-[120px]">{t("status")}</TableHead>
              <TableHead className="min-w-[140px] text-right">{t("total")}</TableHead>
              <TableHead className="w-28 text-right">{t("actions")}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {rows.length ? (
              rows.map((po) => (
                <TableRow key={po.id}>
                  <TableCell className="font-medium">{po.poNumber}</TableCell>
                  <TableCell>{po.supplier}</TableCell>
                  <TableCell>{po.date}</TableCell>
                  <TableCell>{po.expected}</TableCell>
                  <TableCell>
                    <StatusPill status={po.status} t={t} />
                  </TableCell>
                  <TableCell className="text-right tabular-nums">
                    {po.total.toFixed(2)} {po.currency}
                  </TableCell>
                  <TableCell className="text-right space-x-2">
                   
                    <Button asChild size="sm" variant="outline">
                      <Link href={`/inventory/purchase-orders/${po.id}`}>{t("view")}</Link>
                    </Button>
                    <Button asChild size="sm" variant="outline">
                      <Link href={`/inventory/purchase-orders/${po.id}/edit`}>{t("edit")}</Link>
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => {
                        // TODO: We add our status to update the logic (API call or state chang here)
                        console.log(`Mark PO ${po.id} as received`);
                      }}
                    >
                      {t("receive")}
                    </Button>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={7} className="text-center text-muted-foreground">
                  {t("noData")}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  );
}

function StatusPill({
  status,
  t,
}: {
  status: "draft" | "sent" | "partial" | "closed";
  t: (k: string) => string;
}) {
  const map: Record<string, string> = {
    draft: "bg-muted text-foreground",
    sent: "bg-blue-100 text-blue-800 dark:bg-blue-900/40 dark:text-blue-300",
    partial: "bg-amber-100 text-amber-800 dark:bg-amber-900/40 dark:text-amber-300",
    closed: "bg-emerald-100 text-emerald-800 dark:bg-emerald-900/40 dark:text-emerald-300",
  };
  return (
    <span className={`inline-flex items-center rounded px-2 py-0.5 text-xs font-medium ${map[status]}`}>
      {t(`status${status.charAt(0).toUpperCase() + status.slice(1)}`)}
    </span>
  );
}
