"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select"
import { PrintButton } from "../_components/report-helpers"

type AdjustmentRow = {
  date: string
  item: string
  unit: string
  type: string
  newQty: number
  current: number
  by: string
  reason: string
}

const items = [
  { id: "i1", name: "Gloves" },
  { id: "i2", name: "Syringes" },
]

const seed: AdjustmentRow[] = [
  {
    date: "2025-07-10",
    item: "Gloves",
    unit: "BOX",
    type: "Manual",
    newQty: 1,
    current: 0,
    by: "User A",
    reason: "Other",
  },
]

export default function StockAdjustmentReport() {
  const t = useTranslations("Reports")
  const tCommon = useTranslations("common")
  const [item, setItem] = React.useState("all")
  const [from, setFrom] = React.useState("")
  const [to, setTo] = React.useState("")
  const [search, setSearch] = React.useState("")

  const rows = React.useMemo(() => seed.filter(r =>
    (item === "all" || r.item === item) &&
    (!from || r.date >= from) &&
    (!to || r.date <= to) &&
    (!search || Object.values(r).some(val =>
      val?.toString().toLowerCase().includes(search.toLowerCase())
    ))
  ), [item, from, to, search])

  const reset = () => {
    setItem("all")
    setFrom("")
    setTo("")
    setSearch("")
  }

  const downloadCSV = () => {
    const headers = [
      t("adjustment.columns.date"),
      t("adjustment.columns.item"),
      t("adjustment.columns.unit"),
      t("adjustment.columns.type"),
      t("adjustment.columns.newQty"),
      t("adjustment.columns.current"),
      t("adjustment.columns.by"),
      t("adjustment.columns.reason"),
    ]

    const csvRows = [
      headers.join(","),
      ...rows.map(r => [
        r.date,
        r.item,
        r.unit,
        r.type,
        r.newQty,
        r.current,
        r.by,
        r.reason,
      ].map(val => `"${val}"`).join(","))
    ].join("\n")

    const blob = new Blob([csvRows], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "stock_adjustment_report.csv"
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">
          {t("adjustment.title", { default: "Stock Adjustment Report" })}
        </h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print", { default: "Print" })} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export", { default: "Export CSV" })}
          </Button>
        </div>
      </div>


      {/* Filters */}
      <div className="print:hidden grid grid-cols-12 gap-4 items-end">
        <div className="col-span-12 sm:col-span-2">
          <label className="text-sm font-medium">{t("common.from")}</label>
          <Input
            type="date"
            className="h-9 text-sm"
            value={from}
            onChange={e => setFrom(e.target.value)}
          />
        </div>

        <div className="col-span-12 sm:col-span-2">
          <label className="text-sm font-medium">{t("common.to")}</label>
          <Input
            type="date"
            className="h-9 text-sm"
            value={to}
            onChange={e => setTo(e.target.value)}
          />
        </div>

        <div className="col-span-12 sm:col-span-3">
          <label className="text-sm font-medium">{t("common.item")}</label>
          <Select value={item} onValueChange={setItem}>
            <SelectTrigger className="h-9 text-sm w-full">
              <SelectValue placeholder={t("common.item")} />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all")}</SelectItem>
              {items.map(i => (
                <SelectItem key={i.id} value={i.name}>
                  {i.name}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        {/* Search + Reset aligned right */}
        <div className="col-span-12 sm:col-span-5 flex justify-end gap-2">
          <Input
            placeholder={t("searchPlaceholder", { default: "Search..." })}
            value={search}
            onChange={e => setSearch(e.target.value)}
            className="h-9 text-sm w-full sm:w-64"
          />
          <Button
            variant="outline"
            className="h-9 text-sm"
            onClick={reset}
          >
            {t("common.reset")}
          </Button>
        </div>
      </div>

      {/* Print summary */}
      <div className="hidden print:block text-sm text-muted-foreground">
        {t("common.from")}: {from || "—"};&nbsp;
        {t("common.to")}: {to || "—"};&nbsp;
        {t("common.item")}: {item === "all" ? t("common.all") : item}
      </div>

      {/* Table */}
      <div className="w-full border rounded-md overflow-x-auto">
        <table className="table-auto min-w-[900px] w-full">
          <thead className="bg-muted">
            <tr>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.date", { default: "Date" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.item", { default: "Item" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.unit", { default: "Unit" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.type", { default: "Adjustment Type" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.newQty", { default: "New Qty" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.current", { default: "Current Stock" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.by", { default: "Adjusted By" })}
              </th>
              <th className="px-3 py-2 text-left text-sm font-medium">
                {t("adjustment.columns.reason", { default: "Reason" })}
              </th>
            </tr>
          </thead>
          <tbody>
            {rows.length ? (
              rows.map((r, idx) => (
                <tr key={idx} className="border-t">
                  <td className="px-3 py-2 text-sm">{r.date}</td>
                  <td className="px-3 py-2 text-sm">{r.item}</td>
                  <td className="px-3 py-2 text-sm">{r.unit}</td>
                  <td className="px-3 py-2 text-sm">{r.type}</td>
                  <td className="px-3 py-2 text-sm">{r.newQty}</td>
                  <td className="px-3 py-2 text-sm">{r.current}</td>
                  <td className="px-3 py-2 text-sm">{r.by}</td>
                  <td className="px-3 py-2 text-sm">{r.reason}</td>
                </tr>
              ))
            ) : (
              <tr>
                <td
                  colSpan={8}
                  className="text-center text-muted-foreground px-3 py-4"
                >
                  {t("common.noData", { default: "No records found" })}
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
    </div>
  )
}
