"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from "@/components/ui/select"
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from "@/components/ui/table"
import { PrintButton } from "../_components/report-helpers"
import { Button } from "@/components/ui/button"

type Lot = { itemId: string; item: string; unit: string; received: string; qty: number }
type RowAgg = { item: string; unit: string; d0: number; d1: number; d2: number; d3: number }

const items = [
  { id: "i1", name: "Gloves", unit: "BOX" },
  { id: "i2", name: "Syringes", unit: "PK" }
]

const lots: Lot[] = [
  { itemId: "i1", item: "Gloves", unit: "BOX", received: "2025-07-05", qty: 30 },
  { itemId: "i1", item: "Gloves", unit: "BOX", received: "2025-06-20", qty: 50 },
  { itemId: "i2", item: "Syringes", unit: "PK", received: "2025-05-25", qty: 100 }
]

function daysSince(d: string) {
  return Math.floor((Date.now() - new Date(d + "T00:00:00").getTime()) / (1000 * 60 * 60 * 24))
}

export default function AgingReport() {
  const t = useTranslations("Reports")
  const [item, setItem] = React.useState<string>("all")

  const reset = () => setItem("all")

  const filtered = lots.filter(l => item === "all" || l.itemId === item)

  const rows: RowAgg[] = Object.values(
    filtered.reduce<Record<string, RowAgg>>((acc, l) => {
      const days = daysSince(l.received)
      const b0 = days <= 30 ? l.qty : 0
      const b1 = days > 30 && days <= 60 ? l.qty : 0
      const b2 = days > 60 && days <= 90 ? l.qty : 0
      const b3 = days > 90 ? l.qty : 0
      const key = l.itemId
      if (!acc[key]) acc[key] = { item: l.item, unit: l.unit, d0: 0, d1: 0, d2: 0, d3: 0 }
      acc[key].d0 += b0; acc[key].d1 += b1; acc[key].d2 += b2; acc[key].d3 += b3
      return acc
    }, {})
  )

  const downloadCSV = () => {
    const headers = [
      t("aging.columns.name"),
      t("aging.columns.unit"),
      t("aging.columns.d0_30"),
      t("aging.columns.d31_60"),
      t("aging.columns.d61_90"),
      t("aging.columns.d90plus"),
      t("aging.columns.total")
    ]
    const csvRows = rows.map(r => {
      const total = r.d0 + r.d1 + r.d2 + r.d3
      return [r.item, r.unit, r.d0, r.d1, r.d2, r.d3, total].map(val => `"${val}"`).join(",")
    })
    const csv = [headers.join(","), ...csvRows].join("\n")
    const blob = new Blob([csv], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "aging_report.csv"
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">{t("aging.title")}</h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print")} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export")}
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="print:hidden flex flex-wrap gap-4 items-end">
        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("aging.allItems")}</label>
          <Select value={item} onValueChange={setItem}>
            <SelectTrigger className="h-9 w-[200px]">
              <SelectValue placeholder={t("common.selectItem")} />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all")}</SelectItem>
              {items.map(i => (
                <SelectItem key={i.id} value={i.id}>{i.name}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="ml-auto">
          <button
            type="button"
            onClick={reset}
            className="rounded-md border px-3 py-2 text-sm hover:bg-muted"
          >
            {t("common.reset")}
          </button>
        </div>
      </div>

      {/* Print-only summary */}
      <div className="hidden print:block text-sm text-muted-foreground">
        {t("aging.allItems")}: {item === "all" ? t("common.all") : items.find(i => i.id === item)?.name}
      </div>

      {/* Table */}
      <div className="overflow-hidden rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>{t("aging.columns.name")}</TableHead>
              <TableHead>{t("aging.columns.unit")}</TableHead>
              <TableHead>{t("aging.columns.d0_30")}</TableHead>
              <TableHead>{t("aging.columns.d31_60")}</TableHead>
              <TableHead>{t("aging.columns.d61_90")}</TableHead>
              <TableHead>{t("aging.columns.d90plus")}</TableHead>
              <TableHead>{t("aging.columns.total")}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {rows.length ? rows.map((r, idx) => {
              const total = r.d0 + r.d1 + r.d2 + r.d3
              return (
                <TableRow key={idx}>
                  <TableCell>{r.item}</TableCell>
                  <TableCell>{r.unit}</TableCell>
                  <TableCell>{r.d0}</TableCell>
                  <TableCell>{r.d1}</TableCell>
                  <TableCell>{r.d2}</TableCell>
                  <TableCell>{r.d3}</TableCell>
                  <TableCell className="font-medium">{total}</TableCell>
                </TableRow>
              )
            }) : (
              <TableRow>
                <TableCell colSpan={7} className="text-center text-muted-foreground">
                  {t("common.noData")}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}
