"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select"
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell,
} from "@/components/ui/table"
import { PrintButton } from "../_components/report-helpers"
import { Button } from "@/components/ui/button"

type Ledger = {
  date: string
  itemId: string
  item: string
  type: "IN" | "OUT"
  ref: string
  qty: number
}

const items = [
  { id: "i1", name: "Gloves" },
  { id: "i2", name: "Syringes" },
]
const seed: Ledger[] = [
  { date: "2025-07-01", itemId: "i1", item: "Gloves", type: "IN", ref: "B-0001", qty: 100 },
  { date: "2025-07-10", itemId: "i1", item: "Gloves", type: "OUT", ref: "USG-12", qty: 20 },
  { date: "2025-07-15", itemId: "i1", item: "Gloves", type: "OUT", ref: "USG‑18", qty: 10 },
]

export default function StockLedgerReport() {
  const t = useTranslations("Reports")
  const tCommon = useTranslations("common")

  const [from, setFrom] = React.useState("")
  const [to, setTo] = React.useState("")
  const [item, setItem] = React.useState<string>("all")
  const [txType, setTxType] = React.useState<"ALL" | "IN" | "OUT">("ALL")

  const rows = seed
    .filter(r => (item === "all" || r.itemId === item))
    .filter(r => (!from || r.date >= from) && (!to || r.date <= to))
    .filter(r => (txType === "ALL" ? true : r.type === txType))
    .sort((a, b) => a.date.localeCompare(b.date))

  let balance = 0
  const computed = rows.map(r => {
    balance += r.type === "IN" ? r.qty : -r.qty
    return { ...r, in: r.type === "IN" ? r.qty : 0, out: r.type === "OUT" ? r.qty : 0, balance }
  })

  const reset = () => { setFrom(""); setTo(""); setItem("all"); setTxType("ALL") }

  const downloadCSV = () => {
    const headers = [
      t("ledger.columns.date", { default: "Date" }),
      t("ledger.columns.item", { default: "Item" }),
      t("ledger.columns.type", { default: "Type" }),
      t("ledger.columns.reference", { default: "Reference" }),
      t("ledger.columns.qtyIn", { default: "Qty In" }),
      t("ledger.columns.qtyOut", { default: "Qty Out" }),
      t("ledger.columns.balance", { default: "Balance" }),
    ]

    const csvRows = [
      headers.join(","),
      ...computed.map(r =>
        [`"${r.date}"`, `"${r.item}"`, `"${r.type}"`, `"${r.ref}"`, `"${r.in}"`, `"${r.out}"`, `"${r.balance}"`].join(",")
      )
    ].join("\n")

    const blob = new Blob([csvRows], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "stock_ledger_report.csv"
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">
          {t("ledger.title", { default: "Stock Ledger Report" })}
        </h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print", { default: "Print" })} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export", { default: "Export CSV" })}
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="print:hidden flex flex-wrap items-end gap-4">
        <div className="flex flex-col gap-1">
          <label htmlFor="from" className="text-sm font-medium">{t("common.from", { default: "From" })}</label>
          <Input id="from" className="h-9 w-[160px]" type="date" value={from} onChange={e => setFrom(e.target.value)} />
        </div>

        <div className="flex flex-col gap-1">
          <label htmlFor="to" className="text-sm font-medium">{t("common.to", { default: "To" })}</label>
          <Input id="to" className="h-9 w-[160px]" type="date" value={to} onChange={e => setTo(e.target.value)} />
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("common.item", { default: "Item" })}</label>
          <Select value={item} onValueChange={setItem}>
            <SelectTrigger className="h-9 w-[220px]">
              <SelectValue placeholder="All" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All</SelectItem>
              {items.map(i => <SelectItem key={i.id} value={i.id}>{i.name}</SelectItem>)}
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("ledger.columns.type", { default: "Type" })}</label>
          <Select value={txType} onValueChange={v => setTxType(v as "ALL" | "IN" | "OUT")}>
            <SelectTrigger className="h-9 w-[160px]">
              <SelectValue placeholder="All" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="ALL">All</SelectItem>
              <SelectItem value="IN">IN</SelectItem>
              <SelectItem value="OUT">OUT</SelectItem>
            </SelectContent>
          </Select>
        </div>

        <div className="ml-auto">
          <button type="button" onClick={reset} className="h-9 rounded-md border px-3 text-sm hover:bg-muted">
            {t("common.reset", { default: "Reset" })}
          </button>
        </div>
      </div>

      {/* Print filter summary */}
      <div className="hidden print:block text-sm text-muted-foreground">
        Filters: From: {from || "—"}; To: {to || "—"}; Item: {item === "all" ? "All" : (items.find(i => i.id === item)?.name || "—")}; Type: {txType}
      </div>

      {/* Table */}
      <div className="overflow-hidden rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>{t("ledger.columns.date", { default: "Date" })}</TableHead>
              <TableHead>{t("ledger.columns.item", { default: "Item" })}</TableHead>
              <TableHead>{t("ledger.columns.type", { default: "Type" })}</TableHead>
              <TableHead>{t("ledger.columns.reference", { default: "Reference" })}</TableHead>
              <TableHead>{t("ledger.columns.qtyIn", { default: "Qty In" })}</TableHead>
              <TableHead>{t("ledger.columns.qtyOut", { default: "Qty Out" })}</TableHead>
              <TableHead>{t("ledger.columns.balance", { default: "Balance" })}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {computed.length ? computed.map((r, idx) => (
              <TableRow key={idx}>
                <TableCell>{r.date}</TableCell>
                <TableCell>{r.item}</TableCell>
                <TableCell>{r.type}</TableCell>
                <TableCell>{r.ref}</TableCell>
                <TableCell className="tabular-nums">{r.in || ""}</TableCell>
                <TableCell className="tabular-nums">{r.out || ""}</TableCell>
                <TableCell className="tabular-nums">{r.balance}</TableCell>
              </TableRow>
            )) : (
              <TableRow>
                <TableCell colSpan={7} className="text-center text-muted-foreground">
                  {t("common.noMatch", { default: "No matching records" })}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}
