"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select"
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell,
} from "@/components/ui/table"
import { PrintButton } from "../_components/report-helpers"
import { Button } from "@/components/ui/button"

type Item = { id: string; name: string }
type UsageRecord = { date: string; itemId: string; item: string; usage: number }

const items: Item[] = [
  { id: "i1", name: "Gloves" },
  { id: "i2", name: "Syringes" },
]

const seed: UsageRecord[] = [
  // Sample data or leave empty
]

export default function ItemUsageReport() {
  const t = useTranslations("Reports")
  const [item, setItem] = React.useState("all")
  const [from, setFrom] = React.useState("")
  const [to, setTo] = React.useState("")
  const [filtered, setFiltered] = React.useState<UsageRecord[]>([])

  function filterData() {
    const results = seed.filter(r =>
      (item === "all" || r.itemId === item) &&
      (!from || r.date >= from) &&
      (!to || r.date <= to)
    )
    setFiltered(results)
  }

  function downloadCSV() {
    const headers = [
      t("usage.columns.date", { default: "Date" }),
      t("usage.columns.item", { default: "Item" }),
      t("usage.columns.usage", { default: "Usage" }),
    ]
    const csvRows = [
      headers.join(","),
      ...filtered.map(r => `"${r.date}","${r.item}",${r.usage}`)
    ].join("\n")

    const blob = new Blob([csvRows], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "item_usage_report.csv"
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">
          {t("itemUsage.title", { default: "Item Usage Report" })}
        </h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print", { default: "Print" })} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export", { default: "Export CSV" })}
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="print:hidden flex flex-wrap items-end gap-4">
        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("common.item", { default: "Item" })}</label>
          <Select value={item} onValueChange={setItem}>
            <SelectTrigger className="h-9 w-40 text-sm">
              <SelectValue placeholder={t("common.item", { default: "Item" })} />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all", { default: "All" })}</SelectItem>
              {items.map(i => (
                <SelectItem key={i.id} value={i.id}>{i.name}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("common.from", { default: "From" })}</label>
          <Input
            type="date"
            className="h-9 w-[160px] text-sm"
            value={from}
            onChange={e => setFrom(e.target.value)}
          />
        </div>
        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("common.to", { default: "To" })}</label>
          <Input
            type="date"
            className="h-9 w-[160px] text-sm"
            value={to}
            onChange={e => setTo(e.target.value)}
          />
        </div>

        <Button className="h-9 mt-5 text-sm" onClick={filterData}>
          {t("common.filter", { default: "Filter" })}
        </Button>
      </div>

      {/* Print Filter Summary */}
      <div className="hidden print:block text-sm text-muted-foreground">
        {t("filters.summary", { default: "Filters" })}:&nbsp;
        {t("common.item")}: {item === "all" ? t("common.all") : items.find(i => i.id === item)?.name};&nbsp;
        {t("common.from")}: {from || "—"};&nbsp;
        {t("common.to")}: {to || "—"};
      </div>

      {/* Data or Empty Message */}
      {filtered.length === 0 ? (
        <div className="rounded-md border bg-muted/30 p-4 text-center text-red-600">
          {t("common.noDataFound", { default: "No data found. Try different dates." })}
        </div>
      ) : (
        <div className="overflow-hidden rounded-md border">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>{t("usage.columns.date", { default: "Date" })}</TableHead>
                <TableHead>{t("usage.columns.item", { default: "Item" })}</TableHead>
                <TableHead>{t("usage.columns.usage", { default: "Usage" })}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filtered.map((r, idx) => (
                <TableRow key={idx}>
                  <TableCell>{r.date}</TableCell>
                  <TableCell>{r.item}</TableCell>
                  <TableCell>{r.usage}</TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </div>
      )}
    </div>
  )
}
