"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Check, X, Wrench } from "lucide-react"
import { Button } from "@/components/ui/button"
import StockRequestTable from "@/components/stock/StockRequestTable"
import type { StockRequestLine } from "@/components/stock/stock-request-types"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { useRouter } from "next/navigation"

export default function StockRequestReviewPage() {
  const t = useTranslations()
  const router = useRouter()

  const [meta] = React.useState({
    requestNumber: "SR-00045",
    fromLocation: "Main Store",
    toLocation: "OPD",
    requestedBy: "Nurse Mary",
    createdAt: new Date().toISOString(),
    notes: "Urgent for morning shift",
  })

  const [lines, setLines] = React.useState<StockRequestLine[]>([
    {
      id: crypto.randomUUID(),
      item: "Syringe 5ml",
      unit: "box", // ✅ NEW FIELD
      requestedQty: 50,
      adjustedQty: 40,
    },
    {
      id: crypto.randomUUID(),
      item: "Alcohol Swab",
      unit: "pack", // ✅ NEW FIELD
      requestedQty: 10,
      adjustedQty: 10,
    },
  ])

  function updateLine(id: string, patch: Partial<StockRequestLine>) {
    setLines((prev) => prev.map((l) => (l.id === id ? { ...l, ...patch } : l)))
  }

  function approve() {
    console.log("APPROVED", { lines })
    router.back()
  }

  function reject() {
    console.log("REJECTED")
    router.back()
  }

  function adjust() {
    console.log("ADJUSTED", { lines })
  }

  return (
    <div className="w-full p-4 md:p-6 lg:p-8 space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-semibold">{t("stockRequests.review")}</h1>
        <div className="flex gap-2">
          <Button variant="outline" onClick={reject}>
            <X className="mr-2 h-4 w-4" />
            {t("stockRequests.reject")}
          </Button>
          <Button variant="outline" onClick={adjust}>
            <Wrench className="mr-2 h-4 w-4" />
            {t("stockRequests.adjust")}
          </Button>
          <Button onClick={approve}>
            <Check className="mr-2 h-4 w-4" />
            {t("stockRequests.approveSend")}
          </Button>
        </div>
      </div>

      {/* Request metadata */}
      <div className="rounded-md border bg-card p-4">
        <div className="grid grid-cols-12 gap-3">
          <Field label={t("stockRequests.fromLocation")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={meta.fromLocation} readOnly />
          </Field>
          <Field label={t("stockRequests.toLocation")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={meta.toLocation} readOnly />
          </Field>
          <Field label={t("stockRequests.requestedBy")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={meta.requestedBy} readOnly />
          </Field>
          <Field label={t("stockRequests.notes")} className="col-span-12">
            <Textarea value={meta.notes} readOnly />
          </Field>
        </div>
      </div>

      {/* Table with editable adjustedQty */}
      <StockRequestTable
        lines={lines}
        updateLine={updateLine}
        editable={false}
        showAdjusted={true}
      />
    </div>
  )
}

function Field({
  label,
  className,
  children,
}: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={className}>
      <label className="mb-1 block text-xs font-medium text-muted-foreground">{label}</label>
      {children}
    </div>
  )
}
